#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "definitions.h"
#include "vdw_radii.h"
#include "geom.h"
#include "connect.h"
#include "scor.h"
#include "ap_class.h"
#include "lenjon.h"

float rotsco(int nlig, int ntot, int flag[], int asy[], int maxcon, int **con, 
             int nbond, struct bond bl[], float **xyz)
/*   Defines the score for lipophilic contacts:

     Eldridge M.D. et al, J. Comput.-Aided Mol. Des., 11 (1997), 425-445.
    
     p. 431.          
 */
{
  register int i, j, k;
  float rsco = 0., tsco;
  int idx_a, idx_b, nrot = 0;
  int na, nb;
  int *va, *vb;
  ap_hash close_p(1000);
  
  // calculate van der Waal contacts once, store for fast retrival
  for(i=0; i<nbond; ++i) {
    if(asy[i] == _H_) continue;
    for(k=nlig; k<ntot; ++k) {
      if(asy[k] == _H_) continue;
      if(vdw_contact(i, k, asy, xyz)) close_p.store(i, k);
    }
  }

  va = (int *) malloc(nlig * sizeof(int));
  vb = (int *) malloc(nlig * sizeof(int));
  
  for(i=0; i<nbond; ++i) {
    if(!(bl[i].rota)) continue;
    /* fill lists of atoms at both ends of the bond */
    part_bond(bl[i].aa, bl[i].ab, nlig, maxcon, con, &na, &nb, va, vb);
    /* remove H-atoms */
    idx_a = cmprss(na, va, asy, _H_);
    idx_b = cmprss(nb, vb, asy, _H_);
    na = idx_a; nb = idx_b;
    /* check for van der Waals contacts */
    idx_a = idx_b = 0;
    for(k=nlig; k<ntot; ++k) {
      if(asy[k] == _H_) continue;
      for(j=0; j<na && !idx_a; ++j) idx_a += close_p.retrieve(va[j]-1, k);
      if(idx_a && idx_b) break;
      for(j=0; j<nb && !idx_b; ++j) idx_b += close_p.retrieve(vb[j]-1, k);
      if(idx_a && idx_b) break;
    }
    /* the bond is not blocked */
    if(!idx_a || !idx_b) continue;
    ++nrot;
    idx_a = idx_b = 0;
    /* calculate percentage of polar atoms */
    for(j=0; j<na; ++j) if(flag[va[j]-1] != Non_Polar) ++idx_a;
    for(j=0; j<nb; ++j) if(flag[vb[j]-1] != Non_Polar) ++idx_b;
    /* add to the list */
    rsco += (0.5*(((float) idx_a)/na + ((float) idx_b)/nb));
  } /* i */
  
  free(va); free(vb);

  tsco = nrot ? 1. + (1. - 1./nrot)*rsco : 0.;
  
  return tsco;
}

float lipsco(int nlig, int ntot, int flag[], int asy[], float **xyz)
/*   Defines the score for lipophilic contacts:

     Eldridge M.D. et al, J. Comput.-Aided Mol. Des., 11 (1997), 425-445.
    
     p. 431.
 */
{
  register int i, j;
  float dsco, lsco = 0.;
  float vdw_lig, vdw_rec;
  float R1, R2;
  float d2;

  for(i=0; i<nlig; ++i) {
    if(asy[i] == _H_) continue;
    if(flag[i] != Lipo) continue;
    vdw_lig = vdW_Radii[asy[i]];
    for(j=nlig; j<ntot; ++j) {
      if(asy[j] == _H_) continue;
      if(flag[j] != Lipo) continue;
      vdw_rec = vdW_Radii[asy[j]];
      R1 = vdw_lig + vdw_rec + 0.5;
      R2 = R1 + 3.;
      d2 = dsq(i, j, xyz);
      if(d2 < R1*R1) dsco = 1.;
      else
      if(d2 > R2*R2) dsco = 0.;
      else {
        dsco = 1. - (sqrt(d2) - R1)/(R2 - R1);
      }
      lsco += dsco;
    }  /* j */
  }  /* i */

  return lsco;
}

float burp(int nlig, int ntot, int flag[], int asy[], float **xyz)
/*   count the buried polar atoms
 */
{
  register int i, j;
  float dsco, lsco = 0.;
  float vdw_lig, vdw_rec;
  float R1, R2;
  float d2;
  int i_lipo, j_lipo;

  for(i=0; i<nlig; ++i) {
    if(asy[i] == _H_) continue;
    i_lipo = (flag[i] == Lipo ? 1 : 0);
    vdw_lig = vdW_Radii[asy[i]];
    for(j=nlig; j<ntot; ++j) {
      if(asy[j] == _H_) continue;
      j_lipo = (flag[j] == Lipo ? 1 : 0);
      if(i_lipo && j_lipo) continue;
      vdw_rec = vdW_Radii[asy[j]];
      R1 = vdw_lig + vdw_rec + 0.5;
      R2 = R1 + 3.;
      d2 = dsq(i, j, xyz);
      if(d2 < R1*R1) dsco = 1.;
      else
      if(d2 > R2*R2) dsco = 0.;
      else {
        dsco = 1. - (sqrt(d2) - R1)/(R2 - R1);
      }
      lsco += dsco;
    }  /* j */
  }  /* i */

  return lsco;
}

float hb_score(int nlig, int ntot, int flag[], int **con, float **xyz)
/*
    Boehm's scoring function
 */
{
  register int i, j, k;
  float score = 0.;
  int is_accept;

  for(i=0; i<nlig; ++i) {
    if(flag[i] != Donor) continue;
    k = con[i][0]-1;
    for(j=nlig; j<ntot; ++j) {
      is_accept = flag[j] == Accept || flag[j] == Don_Acc;
      if(!is_accept) continue;
      score += boehm(dsq(i, j, xyz), ang(k, i, j, xyz));
    }  /* j */
  }  /* i */

  for(j=0; j<nlig; ++j) {
    is_accept = flag[j] == Accept || flag[j] == Don_Acc;
    if(!is_accept) continue;
    for(i=nlig; i<ntot; ++i) {
      if(flag[i] != Donor) continue;
      k = con[i][0]-1;
      score += boehm(dsq(i, j, xyz), ang(k, i, j, xyz));
    }  /* i */
  }  /* j */

  return score;
}

/*********************************************************************
 *********************************************************************/

float boehm(float d2, float alf)
/*  Defines the HB score in terms of the X-A distance and
    the XH-A angle

   
 Score  ^
        |       1.6     2.1
      1.|_ _ _ _ _________
        |       /|   |   |\
        |      / |   |   | \
        |     /  |   |   |  \
      0.--------------------------------------->
            1.3     1.85    2.5    H-X Distance


       180     150
        |sc.=1 /
        |     /
        |    / 
        |   /  
        |  /  score =
        | / 1-(dA-30)/50
        |/   
        |------ 100
        |
        | score=0
        |

 */     
{
  const float RLO0SQ = 1.69;  /* 1.3 * 1.3 */
  const float RHI0SQ = 6.25;  /* 2.5 * 2.5 */
  const float RLO1SQ = 2.56;  /* 1.6 * 1.6 */
  const float RHI1SQ = 4.41;  /* 2.1 * 2.1 */
  const float REQ = 1.85;
  const float DER = 0.25;
  const float ddn = 2.5;      /* 1/0.4 */

  const float adn = 0.02;     /* 1/50 */
  const float ALO = 1.745329; /* 100 * PI / 180 */
  const float AHI = 2.617994; /* 150 * PI / 180 */
  
  float de, dsco, asco;
  
  /* radial contribution */
  if(d2 < RLO0SQ || d2 > RHI0SQ) dsco = 0.;
  else
  if(d2 > RLO1SQ && d2 < RHI1SQ) dsco = 1.;
  else {
    de = fabs(sqrt(d2) - REQ);
    dsco = 1. - (de-DER)*ddn;
  }
  if(dsco < OM6) return 0.;

  /* angular contribution */
  if(alf > AHI) asco = 1.;
  else
  if(alf < ALO) asco = 0.;
  else {
    de = 180. - TODEG(alf);
    asco = 1. - (de - 30)*adn;
  }

  return asco*dsco;
}

int vdw_contact(int aa, int ab, int asy[], float **xyz)
/* contact between two atoms
 */
{
  float d2, R1;

  d2 = dsq(aa, ab, xyz);
  R1 = vdW_Radii[asy[aa]] + vdW_Radii[asy[ab]] + 0.5;

  return d2 <= R1*R1;
}

float flxlip(int nlig, int ntot, int flag[], int asy[], float **xyz)
/*	Defines the score for lipophilic contacts

		Rarey, M et al. J. Mol. Biol. (1996) 261, 470-489.
		p. 473.

 */
{
  register int i, j;
  float dsco, lsco = 0.;
  float vdw_lig, vdw_rec;
  float R1, R2;
  float d2;

  for(i=0; i<nlig; ++i) {
/*  if(asy[i] == _H_) continue; */
    if(flag[i] != Lipo) continue;
    vdw_lig = vdW_Radii[asy[i]];
    for(j=nlig; j<ntot; ++j) {
/*    if(asy[j] == _H_) continue; */
      if(flag[j] != Lipo) continue;
      vdw_rec = vdW_Radii[asy[j]];
      R1 = vdw_lig + vdw_rec + 0.6;
      R2 = R1 + 0.6;
      d2 = dsq(i, j, xyz);
      if(d2 > R2*R2) dsco = 0.;
      else dsco = rarey(d2, R1);
      lsco += dsco;
    }  /* j */
  }  /* i */

  return lsco;
}

float rarey(float d2, float R1)
{
  const float dn0 = 2.5;      /* 1/0.4 */
  const float dn1 = 5;        /* 1/0.2 */

  float de;
  float RLO1SQ = (R1-.2)*(R1-.2);
  float RLO2SQ = (R1-.6)*(R1-.6);
  float RHI1SQ = (R1+.2)*(R1+.2);
  
  if(d2 > RLO1SQ && d2 < RHI1SQ) return 1.;
  else {
    de = sqrt(d2) - R1;
	  if(d2 < RLO1SQ && d2 > RLO2SQ) return (1.+(de+.2)*dn0);
	  else
	  if(d2 > RHI1SQ) return (1.-(de-.2)*dn0);
	  else
	  if(d2 < RLO2SQ) return (de+.6)*dn1;
  }    
  return 0.;
}

void lj_int(int nlig, int ntot, int asy[], float **xyz,
            float *lj_rep, float *lj_att)
/*
    Lennard-Jones repulsion.
    Parameters from "AUTODOCK", results in kcal/mol.
 */
{
  const float LJTHR = 24.;
  lj12 lj12;
  lj6 lj6;

  register int i, j;
  float vdw_lig, vdw_rec, c12 = 0., c6 = 0.;
  float R1;
  float d2;

  *lj_att = *lj_rep = 0.;

  for(i=0; i<nlig; ++i) {
    if(asy[i] == _H_) continue;
    vdw_lig = vdW_Radii[asy[i]];
    for(j=nlig; j<ntot; ++j) {
      if(asy[j] == _H_) continue;
      vdw_rec = vdW_Radii[asy[j]];
      R1 = vdw_lig + vdw_rec + LJTHR;
      d2 = dsq(i, j, xyz);
      if(d2 > R1*R1) continue;
      lj12.retrieve(asy[i], asy[j], c12);
      *lj_rep += lenjon(sqrt(d2), 12, c12);
      lj6.retrieve(asy[i], asy[j], c6);
      *lj_att += lenjon(sqrt(d2), 6, c6);
    }  /* j */
  }  /* i */
}

float lenjon(float ds, int pwr, float cff)
{
  //register int i;
  float x = 1./ds;
  float den = 1.;

  //for(i=0; i<pwr; ++i) den *= x;
  while (pwr!=0) {
		if (pwr%2!=0) den*=x;
		pwr/=2;
		x*=x;
  }

  return cff*den;
}



