#include <stdlib.h>

#include "definitions.h"
#include "connect.h"
#include "prune.h"
#include "hybr.h"
#include "rngblk.h"
#include "ap_class.h"

int mol_count(int nat, int ncon, int **con, int flag[])
/* counts the number of molecules and assigns a molecule
 * number to each atom
 */
{
  register int i;
  int nmol = 1;

  /* loop over atoms */
  for(i=0; i<nat; ++i) {
    if(flag[i] == NULL) {
      /* flag this atom -- and all atoms connected to it */
      flag_rec(i, nat, ncon, con, flag, nmol);
      /* this atom is the first of a new molecule */
      if(flag[i]>0) ++nmol;
    } /* if */
  } /* for(i) */

  /* reset, since one molecule too many has been counted */
  nmol--;
  
  return(nmol);

} /* mol_count() */

void fast_count(int nat, int ref_at, int ncon, int **con, int flag[])
{
  register int i;

  flag_visit(ref_at-1, ncon, con, flag);
  
  for(i=0; i<nat; ++i) if(!flag[i]) flag[i] = 2;
}

void flag_rec(int target, int nat, int ncon, int **con, int flag[],
              int nmol)
/* flags a target atom and, using recursion, all atoms connected
 * to it
 */
{
  register int i, j;
  int mcon = 0;  

  /* disjointed node */
  for(j=0; j<ncon; ++j) if(con[target][j]) ++mcon;
  if(!mcon) {
    flag[target] = -1;
    return;
  }

  /* flag as belonging to the current molecule */
  flag[target] = nmol;

  /* search for all atoms connected to "target" */
  for(i=0; i<nat; ++i) {
    if(flag[i]) continue;
    for(j=0; j<ncon; ++j) {
      if(con[i][j] == target+1) {
        /* flag atom connected to "target", and all others connected to it */
        flag_rec(i, nat, ncon, con, flag, nmol);
        /* break, since two atoms can only be connected once to each other */
        break;
      } /* if */
    } /* for(j) */
  }  /* for(i) */
  
} /* flag_rec() */

void bond_list(int nat, int ncon, int **con, int *nbnd,
               struct bond **bl)
{
  register int i, j;
  int a0, a1;
  
  *bl = (struct bond *) realloc(*bl, 2*nat * sizeof(struct bond));
  
  for(i=0; i<nat; ++i) {
    a0 = i + 1;
    for(j=0; j<ncon; ++j) {
      if((a1 = con[i][j]) == NULL) continue;
      if(a1 < a0) continue;
      ((*bl)[*nbnd]).aa = a0;
      ((*bl)[*nbnd]).ab = a1;
      (*nbnd)++;
    }
  }

  *bl = (struct bond *) realloc(*bl, (*nbnd) * sizeof(struct bond));
}

void bnd2con(int na, int nb, struct bond bl[], int maxcon, int **con)
// build the connection table from the bond list
{
  register int i, j, k;
  int i0, n0, aa, ab;
  // class with binary search of bonds
  ap_search sorted_bond(nb);

  // store and sort the bonds
  for(i = 0; i < nb; ++i) sorted_bond.store(bl[i].aa, bl[i].ab, i);
  sorted_bond.sort();

  for(i = 0; i < na; ++i)
    for(j = 0; j < maxcon; ++j) con[i][j] = 0;

  for(i = 0; i < na; ++i) {
    // first bond with this atom at "i0", with "n0" bonds in total
    i0 = sorted_bond.range(i+1, n0);
    if(i0<0) continue;  // atom not *first* in bond
    for(j = i0; j < i0+n0; ++j) {
      // get second atom in bond
      sorted_bond.retrieve(j, aa, ab);
      // build up the connection table : atom "a" ...
      for(k=0; k<maxcon; ++k) {
        if(con[i][k]) continue;
        con[i][k] = ab;
        break;
      }
      // ... atom "b"
      for(k=0; k<maxcon; ++k) {
        if(con[ab-1][k]) continue;
        con[ab-1][k] = i+1;
        break;
      }
    } // j
  } // i
}

void part_bond(int aa, int ab, int nat, int ncon, int **con,
               int *na, int *nb, int va[], int vb[])
/* Find atoms on both sides of a bond. Return numbers and
   lists of them.
 */             
{
  int *visited;
  register int i;
  
  *na = 0;
  *nb = 0;
  
  visited = (int *) malloc(nat * sizeof(int));
  
  /* find atoms attached to "a" */
  for(i=0; i<nat; ++i) visited[i] = 0;
  visited[ab-1] = 1;
  flag_trav(aa-1, ncon, con, visited, va, na);

  /* find atoms attached to "b" */
  for(i=0; i<nat; ++i) visited[i] = 0;
  visited[aa-1] = 1;
  flag_trav(ab-1, ncon, con, visited, vb, nb);
  
  free(visited);
}
         
void flag_trav(int node, int ncon, int **con,
               int visited[], int trav[], int *i_trav)
/* Detect atoms connected to the first atom called. Return list
   as traversal.
 */
{
  register int k;
  int t0;

  /* add me to the traversal */
  trav[(*i_trav)++] = node+1;
  /* mark as visited */
  visited[node] = 1;

  for(k=0; k<ncon; ++k) {
    if((t0 = con[node][k]) == NULL) continue;
    if(visited[t0-1]) continue;
    flag_trav(t0-1, ncon, con, visited, trav, i_trav);
  }  /* for(k) */
}
         
void flag_visit(int node, int ncon, int **con, int visited[])
/* Detect atoms connected to the first atom called.
 */
{
  register int k;
  int t0;

  /* mark as visited */
  visited[node] = 1;

  for(k=0; k<ncon; ++k) {
    if((t0 = con[node][k]) == NULL) continue;
    if(visited[t0-1]) continue;
    flag_visit(t0-1, ncon, con, visited);
  }  /* for(k) */
}

int cmprss(int na, int va[], int ta[], int trgt)
/* given a list va and a key ta, push elements
   with key value trgt at the end of the list
 */
{
/*
  register int i, j;
  int new_size = na, temp;
  
  for(i=0; i< new_size; ++i) {
    if(ta[va[i]-1] == trgt) {
      --new_size;
      temp = va[i];
      for(j=i; j<new_size; ++j) va[j] = va[j+1];
      va[new_size] = temp;
    }
  }
  
  return new_size;
 */

  int count_up = 0;
  int count_dn = na;
  int *aux;

  aux = (int *) calloc(na, sizeof(int));
 
  for(int i=0; i< na; ++i) {
    if(ta[va[i]-1] != trgt)
      aux[count_up++] = va[i];
    else aux[--count_dn] = va[i];
  }

  for(int j=0; j< na; ++j) va[j] = aux[j];

  free(aux);
 
  return count_up;
}

void get_rot_bond(int nlig, int hyb[], int asy[], int maxcon, int **con,
                  int nbond, struct bond bl[])
{
  register int i, j, k;
  int aa, ab;
  int can_rotate, hcount, fcount;
  int **xcon;
  int *ncon;
  int *flag, *rflag;
  
  ncon = (int *) calloc(nlig, sizeof(int));

  flag = (int *) calloc(nlig, sizeof(int));
  
  xcon = (int **) malloc(nlig * sizeof(int *));
  for(i=0; i<nlig; ++i) xcon[i] = (int *) malloc(maxcon * sizeof(int));

  for(i=0; i<nlig; ++i)
    for(j=0; j<maxcon; ++j) xcon[i][j] = con[i][j];
  
  /* detect and eliminate terminal atoms */
  pruned(nlig, maxcon, xcon, flag);
  
  rflag = flag;
  for(i=0; i<nlig; ++i) rflag[i] = 0;
  
  /* detect rings */
  ring_block(nlig, maxcon, xcon, rflag);

  for(i=0; i<nlig; ++i)
    for(j=0; j<maxcon; ++j) if(con[i][j]) ++ncon[i];
  
  for(i=0; i<nbond; ++i) {
    bl[i].rota = 0;
    aa = bl[i].aa - 1;
    ab = bl[i].ab - 1;
    /* accept sp(2)-sp(3) and sp(3)-sp(3) bonds */
    can_rotate = ((hyb[aa] == SP3 && (hyb[ab] == SP2 || hyb[ab] == SP3)) ||
                  (hyb[ab] == SP3 && (hyb[aa] == SP2 || hyb[aa] == SP3)));
    if(!can_rotate) continue;
    /* accept non-ring pairs or two atoms on different ring blocks */
    can_rotate = ((!rflag[aa] || !rflag[ab]) || (rflag[aa] != rflag[ab]));
    if(!can_rotate) continue;
    /* reject terminal CF(3), CH(3), NH(2), and (+)NH(3) groups */
    hcount = fcount = 0;
    for(j=0; j<maxcon; ++j) {
      if((k=con[aa][j]) == NULL) continue;
      --k;
      if(asy[k] == _H_) ++hcount;
      else
      if(asy[k] == _F_) ++fcount;
    }  /* j */
    can_rotate = ((asy[aa] == _N_ && ((ncon[aa] == 3 && hcount == 2) ||
                                     (ncon[aa] == 4 && hcount == 3))) ||
                  (asy[aa] == _C_ && (hcount == 3 || fcount == 3))) == NULL;
    if(!can_rotate) continue;
    hcount = fcount = 0;
    for(j=0; j<maxcon; ++j) {
      if((k=con[ab][j]) == NULL) continue;
      --k;
      if(asy[k] == _H_) ++hcount;
      else
      if(asy[k] == _F_) ++fcount;
    }  /* j */
    can_rotate = ((asy[ab] == _N_ && ((ncon[ab] == 3 && hcount == 2) ||
                                     (ncon[ab] == 4 && hcount == 3))) ||
                  (asy[ab] == _C_ && (hcount == 3 || fcount == 3))) == NULL;
    if(!can_rotate) continue;
    /* it is a rotatable bond */
    bl[i].rota = 1;
  }
  
  /* free memory */
  free(flag);
  free(ncon);
  for(i=0; i<nlig; ++i) free(xcon[i]);
  free(xcon);
}




