#ifndef _ATOM_CLASS_H_
#define _ATOM_CLASS_H_

// template for hashing
#include "hashtmp.h"
// template for binary search
#include "searchtmp.h"

// Hash table for atoms
class atom_hash;

// atom pair with associated value (float here, could be a class)
class atom_value
{
  protected:
    int a0;
    int a1;
    int a2;
    float value;
  public:
    virtual ~atom_value() {}
    // default value, could not happen in a calculation
    atom_value() : a0(-1), a1(0), a2(0), value(0.) {}
    atom_value(int aa, float v = 0.) : a0(aa), a1(0), a2(0), value(v) {}
    atom_value(int aa, int ab, float v = 0.) : a0(aa), a1(ab), a2(0), value(v) {}
    atom_value(int aa, int ab, int ac, float v = 0.) : a0(aa), a1(ab), a2(ac), value(v) {}
    // boolean operations
    BOOL operator!= (const atom_value& p0) const
    { return ((a0 != p0.a0) || (a1 != p0.a1) || (a2 != p0.a2)); }
    BOOL operator== (const atom_value& p0) const
    { return ((a0 == p0.a0) && (a1 == p0.a1) && (a2 == p0.a2)); }
    // declare a friend class to be able to extract values
    friend atom_hash;
};
    
// Hash table for atoms, with or without specific neighbors
class atom_hash : public hash_table<atom_value>
{
  protected:
    // hash function overloads a purely virtual function
    int hash_func(const atom_value& item) const
    { return (item.a0 * 10 + item.a1 + item.a2) % table_size; }
  public:
    atom_hash(int size = 1000) :
      hash_table<atom_value>(size) {}
    // store for one atom, unspecified neighbors
    virtual BOOL store(int a0, float v) {
      atom_value temp(a0, v);
      return hash_table<atom_value>::store(temp);
    }
    // store for one atom, one neighbor
    virtual BOOL store(int a0, int a1, float v) {
      atom_value temp(a0, a1, v);
      return hash_table<atom_value>::store(temp);
    }
    // store for one atom, two neighbors
    virtual BOOL store(int a0, int a1, int a2, float v) {
      atom_value temp(a0, a1, a2, v);
      return hash_table<atom_value>::store(temp);
    }
    /* look for atom with two neighbors, then one, then
     * unspecified neighbors
     */
    virtual BOOL retrieve(int a0, int a1, int a2, float& v) {
      atom_value in2(a0, a1, a2);
      atom_value in1(a0, a1);
      atom_value in0(a0);
      atom_value out;
      BOOL found;

      found = hash_table<atom_value>::retrieve(in2, out);
      v = out.value;
      if(found) return found;

      found = hash_table<atom_value>::retrieve(in1, out);
      v = out.value;
      if(found) return found;

      found = hash_table<atom_value>::retrieve(in0, out);
      v = out.value;
      return found;
    }
};

#endif




