#ifndef _AP_CLASS_H_
#define _AP_CLASS_H_

// template for hashing
#include "hashtmp.h"
// template for binary search
#include "searchtmp.h"

// Hash table for atom pairs
class ap_hash;
// Binary search
class ap_search;

// atom pair with associated value
template <class real_data>
class ap_tmpl
{
  private:
    int aa;
    int ab;
    real_data value;
  public:
    virtual ~ap_tmpl() {}
    // default value "-1", could not happen in a calculation
    ap_tmpl(int a0 = -1, int a1 = -1, real_data v = 0) :
      aa(a0), ab(a1), value(v) {}
    // extract values : interfaces
    int a0() const { return aa; }
    int a1() const { return ab; }
    real_data x() const { return value; }
};

// derived class where both atoms are matched
class at_pair : public ap_tmpl<float>
{
  public:
    at_pair(int a0 = -1, int a1 = -1, float x = 0.) :
      ap_tmpl<float>(a0, a1, x) {}
    // boolean operations
    BOOL operator< (const at_pair& p0) const
    { return (a0() + a1() < p0.a0() + p0.a1()); }
    BOOL operator!= (const at_pair& p0) const
    { return ((a0() != p0.a0()) || (a1() != p0.a1())); }
    BOOL operator== (const at_pair& p0) const
    { return ((a0() == p0.a0()) && (a1() == p0.a1())); }
};

// derived class where only the first atom is matched
class at_first : public ap_tmpl<float>
{
  public:
    at_first(int a0 = -1, int a1 = -1) : ap_tmpl<float>(a0, a1) {}
    // boolean operations : only first element matched
    BOOL operator< (const at_first& p0) const
    { return (a0() < p0.a0()); }
    BOOL operator!= (const at_first& p0) const
    { return (a0() != p0.a0()); }
    BOOL operator== (const at_first& p0) const
    { return (a0() == p0.a0()); }
};

// Hash table for atom pairs
class ap_hash : public hash_table<at_pair>
{
  protected:
    // hash function overloads a purely virtual function
    int hash_func(const at_pair& item) const
    { return (item.a0() * 10 + item.a1()) % table_size; }
  public:
    ap_hash(int size = 1000) :
      hash_table<at_pair>(size) {}
    virtual BOOL store(int a0, int a1, float v = 0) {
      at_pair temp(a0, a1, v);
      return hash_table<at_pair>::store(temp);
    }
    virtual BOOL retrieve(int a0, int a1) {
      at_pair in(a0, a1), out;
      BOOL found = hash_table<at_pair>::retrieve(in, out);
      return found;
    }
    virtual BOOL retrieve(int a0, int a1, float& v) {
      at_pair in(a0, a1), out;
      BOOL found = hash_table<at_pair>::retrieve(in, out);
      v = out.x();
      return found;
    }
};

// Binary search matching only the first atom in a pair
class ap_search : public brsearch<at_first>
{
  public:
    ap_search(int size = 3000) :
      brsearch<at_first>(size) {}
    virtual void store(int a0, int a1, int index)
    {
      at_first temp(a0, a1);
      datalist<at_first>::store(temp, index);
    }
    virtual int range(int a0, int& size) {
      at_first temp(a0);
      return brsearch<at_first>::search(temp, size);
    }
    virtual void retrieve(int index, int& a0, int& a1)
    {
      at_first temp;
      datalist<at_first>::retrieve(temp, index);
      a0 = temp.a0();
      a1 = temp.a1();
    }
};

#endif




