/*
 *
 * geom3d.c
 * Created: 20081223 Jerome Hert
 *  
 */

#include "geom3d.h"

/*
 * +--------------------------------------------------------------------------+
 * | Prototypes                                                               |
 * +--------------------------------------------------------------------------+
 */

static t_transfo *transfo_cos_sin(t_transfo *transfo);
static t_explore *explore_calculate_increment(t_explore *explore);

/*
 * +--------------------------------------------------------------------------+
 * | Generic functions                                                        |
 * +--------------------------------------------------------------------------+
 */

inline double square(double x) {
	return x*x;
}

inline double point_distance(t_point *A, t_point *B) {
	return sqrt( square(A->x - B->x) + square(A->y - B->y) + square(A->z - B->z) );
}

inline double point_distance_origin(t_point *A) {
	return sqrt( square(A->x) + square(A->y) + square(A->z) );
}

inline int distance_discretize(double x) {
	return (int)(x/(PARAM_DISCRETIZATION));
}


/*
 * Test if the X is right from the plan formed by OAB
 * First you calculate the plane formed by the two vectors OA and OB
 * by calculating the vector OAB = OA * OB
 * A is right of that plane if the scalar product between OAB and X
 * is positive.
 */
int isRightFrom( t_point *A, t_point *B, t_point *X) {
	t_point OAB;

	/*
	 * Normally one would calculate the vector OA and OB but since
	 * O if the origin and has coordinates of 0,0,0
	 * OA is A and
	 * OB is B
	 */

	/* calculate OAB = OA * OB */
	OAB.x = A->y * B->z - A->z * B->y;
	OAB.y = A->z * B->x - A->x * B->z;
	OAB.z = A->x * B->y - A->y * B->x;

	/*
	 * The scalar product between two vectors A and B is:
	 * xa * xb + ya * yb + za * zb
	 */

	return ( OAB.x * X->x + OAB.y * X->y + OAB.z * X->z ) >= ZERO;
}

/*
 * +--------------------------------------------------------------------------+
 * | Function for Point                                                       |
 * +--------------------------------------------------------------------------+
 */

t_point *point_initialize(t_point *point) {
	point->x = 0.;
	point->y = 0.;
	point->z = 0.;
	return point;
}

t_point *point_set(t_point *point, double x, double y, double z) {
	point->x = x;
	point->y = y;
	point->z = z;
	return point;
}

t_point *point_copy(t_point *to, t_point *from) {
	to->x = from->x;
	to->y = from->y;
	to->z = from->z;
	return to;
}

t_point	*point_translate( t_point *point_translated, t_point *point, t_point *vector) {
	point_set(point_translated, point->x+vector->x, point->y+vector->y, point->z+vector->z);
	return point_translated;
}

t_point *point_transform( t_point *point, t_transfo *transfo) {
	t_point tmp;
	/* Rotation by x Axis */
	tmp.x	=	point->x;
	tmp.y	=	point->y * transfo->cos.alpha	+	point->z * transfo->sin.alpha;
	tmp.z	=	-point->y * transfo->sin.alpha	+	point->z * transfo->cos.alpha;
	point_copy(point, &tmp);
	/* Rotation by y Axis */
	tmp.x	=	point->x * transfo->cos.beta	-	point->z * transfo->sin.beta;
	tmp.y	=	point->y;
	tmp.z	=	point->x * transfo->sin.beta	+	point->z * transfo->cos.beta;
	point_copy(point, &tmp);
	/* Rotation by z Axis */
	tmp.x	=	point->x * transfo->cos.gamma	+	point->y * transfo->sin.gamma;
	tmp.y	=	-point->x * transfo->sin.gamma	+	point->y * transfo->cos.gamma;
	tmp.z	=	point->z;
	/* Apply the translation */
	point_translate(point, &tmp, &transfo->translation);
	return point;
}

//t_point *point_rotate( t_point *point, t_transfo *transfo) {
//	t_point tmp;
//	/* Rotation by x Axis */
//	tmp.x	=	point->x;
//	tmp.y	=	point->y * transfo->cos.alpha	+	point->z * transfo->sin.alpha;
//	tmp.z	=	-point->y * transfo->sin.alpha	+	point->z * transfo->cos.alpha;
//	point_copy(point, &tmp);
//	/* Rotation by y Axis */
//	tmp.x	=	point->x * transfo->cos.beta	-	point->z * transfo->sin.beta;
//	tmp.y	=	point->y;
//	tmp.z	=	point->x * transfo->sin.beta	+	point->z * transfo->cos.beta;
//	point_copy(point, &tmp);
//	/* Rotation by z Axis */
//	tmp.x	=	point->x * transfo->cos.gamma	+	point->y * transfo->sin.gamma;
//	tmp.y	=	-point->x * transfo->sin.gamma	+	point->y * transfo->cos.gamma;
//	tmp.z	=	point->z;
//	/* Apply the translation */
//	point_copy(point, &tmp);
//	return point;
//}

void point_print(FILE *fh, t_point *point) {
	fprintf(fh, "x: %8.3f\ty: %8.3f\tz: %8.3f\n", point->x, point->y, point->z);
	return;
}

/*
 * +--------------------------------------------------------------------------+
 * | Functions for Angles                                                     |
 * +--------------------------------------------------------------------------+
 */

t_angle *angle_initialize(t_angle *angle) {
	angle->alpha	= 0.;
	angle->beta		= 0.;
	angle->gamma	= 0.;
	return angle;
}

t_angle *angle_set(t_angle *angle, double alpha, double beta, double gamma) {
	angle->alpha	= alpha;
	angle->beta		= beta;
	angle->gamma	= gamma;
	return angle;
}

t_angle *angle_copy(t_angle *to, t_angle *from) {
	to->alpha	= from->alpha;
	to->beta	= from->beta;
	to->gamma	= from->gamma;
	return to;
}

void angle_print(FILE *fh, t_angle *angle) {
	fprintf(fh, "alpha: %8.3f\tbeta: %8.3f\tgamma: %8.3f\n", angle->alpha, angle->beta, angle->gamma);
	return;
}

/*
 * +--------------------------------------------------------------------------+
 * | Functions for Transfo                                                    |
 * +--------------------------------------------------------------------------+
 */

t_transfo *transfo_initialize(t_transfo *transfo) {
	point_initialize(&transfo->translation);
	angle_initialize(&transfo->rotation_angle);
	transfo_cos_sin(transfo);
	return transfo;
}

t_transfo *transfo_set(t_transfo *transfo, t_angle rotation_angle, t_point translation) {
	point_copy(	&transfo->translation,		&translation);
	angle_copy(	&transfo->rotation_angle,	&rotation_angle);
	transfo_cos_sin(transfo);
	return transfo;
}

t_transfo *transfo_copy(t_transfo *to, t_transfo *from) {
	point_copy(&to->translation,	&from->translation);
	angle_copy(&to->rotation_angle,	&from->rotation_angle);
	transfo_cos_sin(to);
	return to;
}

static t_transfo *transfo_cos_sin(t_transfo *transfo) {
	angle_set(	&transfo->cos, cos(transfo->rotation_angle.alpha), cos(transfo->rotation_angle.beta), cos(transfo->rotation_angle.gamma) );
	angle_set(	&transfo->sin, sin(transfo->rotation_angle.alpha), sin(transfo->rotation_angle.beta), sin(transfo->rotation_angle.gamma) );
	return transfo;
}

//int transfo_same_zone(t_transfo *transfo1, t_transfo *transfo2) {
//	int same_zone = FALSE;
//	t_point p1 = {ONE,ONE,ONE}, p2 = {ONE,ONE,ONE};
//	point_rotate(&p1,transfo1);
//	point_rotate(&p2,transfo2);
//	if( point_distance(&p1, &p2) < DISTANCE_MAX_SAME_ZONE )
//		same_zone = TRUE;
//	return same_zone;
//}

void transfo_print(FILE *fh, t_transfo *transfo) {
	fprintf(fh, "Translation\t:\t");
	point_print(fh, &transfo->translation);
	fprintf(fh, "Rotation angle\t:\t");
	angle_print(fh, &transfo->rotation_angle);
	return;
}

/*
 * +--------------------------------------------------------------------------+
 * | Functions for Explore                                                    |
 * +--------------------------------------------------------------------------+
 */

t_explore *explore_initialize(t_explore *explore) {
	explore->rotation_nb			= SITEALIGN_NROTATION_DEFAULT;
	explore->translation_nb			= SITEALIGN_NTRANSLATION_DEFAULT;
	explore->rotation_int			= SITEALIGN_IROTATION_DEFAULT;
	explore->translation_int		= SITEALIGN_ITRANSLATION_DEFAULT;
	explore_calculate_increment(explore);
	return explore;
}

t_explore *explore_set( t_explore *explore, int nrot, int ntrans, double irot, double itrans) {
	explore->rotation_nb		= nrot;
	explore->translation_nb		= ntrans;
	explore->rotation_int		= irot;
	explore->translation_int	= itrans;
	explore_calculate_increment(explore);
	return explore;
}

t_explore *explore_set_nrot( t_explore *explore, int nrot) {
	explore->rotation_nb		= nrot;
	explore_calculate_increment(explore);
	return explore;
}

t_explore *explore_set_ntrans( t_explore *explore, int ntrans) {
	explore->translation_nb	= ntrans;
	explore_calculate_increment(explore);
	return explore;
}

t_explore *explore_set_irot( t_explore *explore, double irot) {
	explore->rotation_int		= irot;
	explore_calculate_increment(explore);
	return explore;
}

t_explore *explore_set_itrans( t_explore *explore, double itrans) {
	explore->translation_int	= itrans;
	explore_calculate_increment(explore);
	return explore;
}

static t_explore *explore_calculate_increment(t_explore *explore) {
//	explore->rotation_increment		= explore->rotation_int  / (explore->rotation_nb - ONE) ;
	explore->rotation_increment		= explore->rotation_int  / explore->rotation_nb ;
	explore->translation_increment	= explore->translation_nb <= ONE ? ONE : explore->translation_int / ( explore->translation_nb - ONE );
	return explore;
}

t_explore *explore_copy(t_explore *to, t_explore *from) {
	to->rotation_nb				= from->rotation_nb;
 	to->translation_nb			= from->translation_nb;
 	to->rotation_int			= from->rotation_int;
 	to->translation_int			= from->translation_int;
 	to->rotation_increment		= from->rotation_increment;
 	to->translation_increment	= from->translation_increment;
 	return to;
}

t_explore *explore_calculate_refinement(t_explore *explore) {
	explore->rotation_int			= explore->rotation_increment		* TWO;
	explore->rotation_nb			= explore->rotation_nb				/ TWO;
	explore->translation_int		= explore->translation_increment	* TWO;
	explore_calculate_increment(explore);
	return explore;
}

void explore_print(FILE *fh, t_explore *explore) {
//	fprintf(fh, "WARNING:\tNumber of rotations\t->\t%d\n", 			explore->rotation_nb);
//	fprintf(fh, "WARNING:\tNumber of translations\t->\t%d\n", 		explore->translation_nb);
//	fprintf(fh, "WARNING:\tIntensity of rotation\t->\t%.3f\n", 		explore->rotation_int);
//	fprintf(fh, "WARNING:\tIntensity of translation\t->\t%6.3f\n",	explore->translation_int);
//	fprintf(fh, "WARNING:\tIncrement of rotations\t->\t%6.3f\n",	explore->rotation_increment);
//	fprintf(fh, "WARNING:\tIncrement of translations\t->\t%6.3f\n",	explore->translation_increment);
	fprintf(fh, "Rop=%g Rn=%d Ri=%g Top=%g Tn=%d Ti=%g\n",
		explore->rotation_int * 180 / PI,
		explore->rotation_nb,
		explore->rotation_increment * 180 / PI,
		explore->translation_int,
		explore->translation_nb,
		explore->translation_increment
	);
	return;
}
