/*
 *
 * protein.c
 * Created: 20081222 Jerome Hert
 *  
 */

#include "protein.h"

/*
 * +--------------------------------------------------------------------------+
 * | Functions for Protein                                                    |
 * +--------------------------------------------------------------------------+
 */
t_protein *protein_initialize(t_protein *protein) {
	protein->file_protein[FALSE]	= '\0';
	protein->file_residues[FALSE]	= '\0';
	protein->size					= ZERO;
	protein->name					= NULL;
	protein->header					= NULL;
	protein->footer					= NULL;
	protein->residue				= NULL;
	point_initialize(&protein->centerofca);
	return protein;
}

t_protein *protein_copy(t_protein *to, t_protein *from) {
	int i;
	/* Just make sure this is not going to be a memory leak
	 * So free all the fields that need to be calloc'ed
	 */
	protein_free(to);
	/* Copy all the fields */
	strncpy(to->file_residues, from->file_residues, MAXCHAR);
	strncpy(to->file_protein, from->file_protein, MAXCHAR);
	to->size = from->size;
	point_copy(&to->centerofca, &from->centerofca);
	/*
	 * Normaly would also copy name, header and footer
	 */
	/* Copy all the residues (and allocate memory for them) */
	if (from->residue != NULL) {
		to->residue = (t_residue *)calloc(to->size, sizeof(t_residue));
		for (i=ZERO;i<to->size;i++)
			residue_copy(&to->residue[i], &from->residue[i]);
	}
	return to;
}

t_protein *protein_copy_ca_cb_cm(t_protein *to, t_protein *from) {
	int i;
	if ( to->size != from->size ) {
		fprintf(stderr, "ERROR (protein_copy_ca_cb_cm): Trying to copy Calphas, Cbetas and Centers of Mass in proteins of different sizes\n");
		exit(ONE);
	}
	point_copy(&to->centerofca, &from->centerofca);
	for (i=ZERO;i<to->size;i++)
		residue_copy_ca_cb_cm(&to->residue[i], &from->residue[i]);	
	return to;
}

t_protein *protein_copy_infos(t_protein *to, t_protein *from) {
	int i;
	if (to != NULL && from != NULL) {
		/* Copy all the fields */
		if (to->file_residues != NULL && from->file_residues != NULL)
			strncpy(to->file_residues, from->file_residues, MAXCHAR);
		if (to->file_protein != NULL && from->file_protein != NULL)
			strncpy(to->file_protein, from->file_protein, MAXCHAR);
		to->size = from->size;
		point_copy(&to->centerofca, &from->centerofca);
		/*
		 * Normaly would also copy name, header and footer
		 */
		/* Copy all the residues (and allocate memory for them) */
		if (to->residue != NULL && from->residue != NULL) {
			for (i=ZERO;i<to->size;i++)
				residue_copy(&to->residue[i], &from->residue[i]);
		}
	}
	return to;
}

t_protein *protein_centerofca(t_protein *protein) {
	int i, j;
	t_point centerofca;
	point_initialize(&centerofca);
	for (i=ZERO;i<protein->size;i++) {
		if (protein->residue[i].type == '?')
			continue;
		j = protein->residue[i].Ca;
		centerofca.x += protein->residue[i].atom[j].coord.x;
		centerofca.y += protein->residue[i].atom[j].coord.y;
		centerofca.z += protein->residue[i].atom[j].coord.z;
	}
	protein->centerofca.x = centerofca.x / protein->size;
	protein->centerofca.y = centerofca.y / protein->size;
	protein->centerofca.z = centerofca.z / protein->size;
	return protein;
}

t_protein *protein_to_origin(t_protein *protein) {
	t_point		translation = {ZERO,ZERO,ZERO};
	t_angle		rotation	= {ZERO,ZERO,ZERO};
	t_transfo	transfo;
	point_set(&translation, -protein->centerofca.x, -protein->centerofca.y, -protein->centerofca.z);
	transfo_set(&transfo, rotation, translation);
	protein_transform(protein,&transfo);
	return protein;
}

t_protein *protein_transform( t_protein *protein, t_transfo *transfo) {
	int i;
	for(i=ZERO;i<protein->size;i++)
		residue_transform(&protein->residue[i], transfo);
	return protein;
}

t_protein *protein_transform_ca_cb_cm( t_protein *protein, t_transfo *transfo) {
	int i;
	for(i=ZERO;i<protein->size;i++)
		residue_transform_ca_cb_cm(&protein->residue[i], transfo);
	return protein;
}

int protein_prepare(t_protein *protein) {
	int i, missing_ca_cb = ZERO;
	for (i=ZERO;i<protein->size;i++)
		missing_ca_cb += residue_prepare(&protein->residue[i]);
	return missing_ca_cb;
}

int protein_prepare_d3_d4(t_protein *protein) {
	int i, missing_ca_cb = ZERO;
	for (i=ZERO;i<protein->size;i++)
		missing_ca_cb += residue_prepare_d3_d4(&protein->residue[i]);
	return missing_ca_cb;
}

void protein_free(t_protein *protein) {
	int i;
	if (protein != NULL) {
		if ( protein->residue != NULL) {
			for (i=ZERO;i<protein->size;i++)
				residue_free( &protein->residue[i] );
			free(protein->residue);
			protein->residue	= NULL;
		}
		if (protein->name != NULL) {
			free(protein->name);
			protein->name	= NULL;
		}
		if (protein->header != NULL) {
			free(protein->header);
			protein->name	= NULL;
		}
		if (protein->footer != NULL) {
			free(protein->footer);
			protein->footer	= NULL;
		}
	}
	return;
}

void protein_print(FILE *fh, t_protein *protein) {
	int i;
	fprintf(fh, "#-----------------------------------------------------------------\n");
	fprintf(fh, "Residue File:\t%s\n", protein->file_residues);
	fprintf(fh, "Protein File:\t%s\n", protein->file_protein);
	fprintf(fh, "Number of Residues:\t%d\n", protein->size);
	fprintf(fh, "Center of Calpha:\t%8.3f\t%8.3f\t%8.3f\n", protein->centerofca.x, protein->centerofca.y, protein->centerofca.z);
	for (i=ZERO;i<protein->size;i++) {
		fprintf(fh, "\t#----------------------\n");
		residue_print(fh, &protein->residue[i]);
	}
	fprintf(fh, "#-----------------------------------------------------------------\n");
	fprintf(fh, "\n");
	return;
}

/*
 * +--------------------------------------------------------------------------+
 * | Function for Residue                                                     |
 * +--------------------------------------------------------------------------+
 */
t_residue *residue_initialize(t_residue *residue) {
	residue->id					= ZERO;
	residue->chain				= '\0';
	residue->type				= '\0';
	residue->type_long[FALSE]	= '\0';
	residue->size				= ZERO;
	residue->atom				= NULL;
	residue->Ca					= MINUSONE;
	residue->Cb					= MINUSONE;
	point_initialize(&residue->centerofmass);
	return residue;
}

t_residue *residue_copy(t_residue *to, t_residue *from) {
	int i;
	/* Just make sure there is not already something assigned to the atom part */
	if (to->atom != NULL)
		residue_free(to);
	/* Copy all the fields */
	strncpy(to->type_long, from->type_long, MAX_RESIDUE_TYPE_LONG);
	to->id			= from->id;
	to->chain		= from->chain;
	to->type		= from->type;
	to->size		= from->size;
	to->Ca			= from->Ca;
	to->Cb			= from->Cb;
	point_copy(&to->centerofmass, &from->centerofmass);
	/* Copy all the atoms (and allocate memory for them first) */
	if (from->atom != NULL) {
		to->atom = (t_atom *)calloc(to->size, sizeof(t_atom));
		for (i=ZERO;i<to->size;i++)
			atom_copy(&to->atom[i], &from->atom[i]);
	}
	return to;
}

t_residue *residue_copy_ca_cb_cm(t_residue *to, t_residue *from) {
	if ( to->id != from->id || to->size != from->size || to->Ca != from->Ca || to->Cb != from->Cb) {
		fprintf(stderr, "ERROR: Trying to copy Calphas, Cbetas and Centers of Mass of different proteins\n");
		exit(ONE);
	}
	atom_copy(	&to->atom[to->Ca],	&from->atom[from->Ca] );
	atom_copy(	&to->atom[to->Cb],	&from->atom[from->Cb] );
	point_copy(	&to->centerofmass,	&from->centerofmass);
	return to;
}

t_residue *residue_populate(t_residue *residue, char *resName, char chainID, int resSeq) {
	strncpy(residue->type_long, resName, MAX_RESIDUE_TYPE_LONG);
	residue->type	= aa_short(residue->type_long);
	residue->chain	= chainID;
	residue->id		= resSeq;
	return residue;
}

t_residue *residue_centerofmass(t_residue *residue) {
	int i, numberofatoms = ZERO;
	t_point centerofmass;
	if (residue->type != '?') {
	point_initialize(&centerofmass);
	for (i=ZERO;i<residue->size;i++) {
			if (
				(		strcmp(residue->atom[i].type_long, "CA")
					&&	strcmp(residue->atom[i].type_long, "C")
					&&	strcmp(residue->atom[i].type_long, "N")
					&&	strcmp(residue->atom[i].type_long, "O")
				)
				||	( residue->type == 'G' && strcmp(residue->atom[i].type_long, "CA") == ZERO )
			) {
				centerofmass.x += residue->atom[i].coord.x;
				centerofmass.y += residue->atom[i].coord.y;
				centerofmass.z += residue->atom[i].coord.z;
				++numberofatoms;
			}
		}
		residue->centerofmass.x = centerofmass.x / numberofatoms;
		residue->centerofmass.y = centerofmass.y / numberofatoms;
		residue->centerofmass.z = centerofmass.z / numberofatoms;
	}
	return residue;
}

t_residue *residue_transform( t_residue *residue, t_transfo *transfo) {
	int i;
	if (residue->type != '?') {
		point_transform(&residue->centerofmass, transfo);
		for(i=ZERO;i<residue->size;i++)
			atom_transform(&residue->atom[i], transfo);
	}
	return residue;
}

t_residue *residue_transform_ca_cb_cm( t_residue *residue, t_transfo *transfo) {
	if (residue->type != '?') {
		atom_transform(		&residue->atom[residue->Ca],	transfo);
		if (residue->type != 'G')
			atom_transform(	&residue->atom[residue->Cb],	transfo);
		point_transform(	&residue->centerofmass, 		transfo);
	}
	return residue;
}

int residue_prepare(t_residue *residue) {
	int i, missing_ca_cb = ZERO;
	if (residue->type != '?') {
		missing_ca_cb = TWO;
		for (i=ZERO;i<residue->size;i++) {
			if ( strcmp(residue->atom[i].type_long, "CA") == ZERO ) {
				residue->Ca = i;
				missing_ca_cb--;
			}
			if ( strcmp(residue->atom[i].type_long, "CB") == ZERO ) {
				residue->Cb = i;
				missing_ca_cb--;
			}
			if (missing_ca_cb == ZERO)
				break;
		}
		if (residue->type == 'G') {
			residue->Cb = residue->Ca;
			missing_ca_cb--;
		}
		if (missing_ca_cb) {
			if (residue->Ca == MINUSONE) {
				residue->Ca = ZERO;
				fprintf(stderr, "WARNING:\tCA not found for residue %c %d. Using first atom read instead.\n", residue->chain, residue->id);
			}
			if (residue->Cb == MINUSONE) {
				residue->Cb = ZERO;
				fprintf(stderr, "WARNING:\tCB not found residue %c %d. Using first atom read instead.\n", residue->chain, residue->id);
			}
		}
		residue_centerofmass(residue);
	}
	return missing_ca_cb;
}

int residue_prepare_d3_d4(t_residue *residue) {
	int i, missing_ca_cb = ZERO;
	char *rot_spe_side_chain_atom = NULL;
	if (residue->type != '?') {
		missing_ca_cb = ONE;
		rot_spe_side_chain_atom = aa_rot_spe_side_chain_atom(residue->type);
		for (i=ZERO;i<residue->size;i++) {
			if ( strcmp(residue->atom[i].type_long, rot_spe_side_chain_atom) == ZERO ) {
				residue->Cb = i;
				missing_ca_cb--;
				break;
			}
		}
		if (missing_ca_cb)
			fprintf(stderr, "WARNING:\tRotamer specific side chain atom %s not found for residue %c %d. Using CB instead.\n", rot_spe_side_chain_atom, residue->type, residue->id);
		free(rot_spe_side_chain_atom);
		residue_centerofmass(residue);
	}
	return missing_ca_cb;
}

void residue_print(FILE *fh, t_residue *residue) {
	int i;
	fprintf(fh, "\tResidue Name:\t%s\n",		residue->type_long);
	fprintf(fh, "\tResidue Type:\t%c\n",		residue->type);
	fprintf(fh, "\tChain Identifier:\t%c\n",	residue->chain);
	fprintf(fh, "\tResidue Number:\t%d\n",		residue->id);
	fprintf(fh, "\tIndex Calpha:\t%d\n",		residue->Ca);
	fprintf(fh, "\tIndex Cbeta:\t%d\n",			residue->Cb);
	fprintf(fh, "\tCenter Of Mass:\t%8.3f\t%8.3f\t%8.3f\n",			residue->centerofmass.x, residue->centerofmass.y, residue->centerofmass.z );
	fprintf(fh, "\tNumber of Atoms:\t%d\n",		residue->size);
	fprintf(fh, "\tList of Atoms:\n");
	for (i=ZERO;i<residue->size;i++) {
		fprintf(fh, "\t\tAtom %3d:", i);
		atom_print(fh, &residue->atom[i]);
	}
	return;
}

void residue_free(t_residue *residue) {
	if (residue != NULL) {
		if (residue->atom != NULL)
			free(residue->atom);
		residue->atom = NULL;
	}
	return;
}

/*
 * +--------------------------------------------------------------------------+
 * | Functions for Atom                                                       |
 * +--------------------------------------------------------------------------+
 */
t_atom *atom_initialize(t_atom *atom) {
	atom->type				= '\0';
	atom->type_long[FALSE]	= '\0';
	point_initialize(&atom->coord);
	return atom;
}

t_atom *atom_copy(t_atom *to, t_atom *from) {
	strncpy(to->type_long, from->type_long, MAX_ATOM_TYPE_LONG);
	to->type = from->type;
	point_copy(&to->coord, &from->coord);
	return to;
}

t_atom *atom_populate(t_atom *atom, char *atom_type, double x, double y, double z, char *element) {
	strncpy(atom->type_long, atom_type, MAX_ATOM_TYPE_LONG);
	point_set(&atom->coord, x, y, z);
	atom->type = element[ZERO];
	return atom;
}

t_atom *atom_transform( t_atom *atom, t_transfo *transfo) {
	point_transform(&atom->coord, transfo);
	return atom;
}

void atom_print(FILE *fh, t_atom *atom) {
	fprintf(fh, "\t\t%s\t%c\t%8.3f\t%8.3f\t%8.3f\n", atom->type_long, atom->type, atom->coord.x, atom->coord.y, atom->coord.z);
	return;
}
